// Defines the entry point for the console application.
//

#include "Includes.h"
#include "Modules.h"
#include "DataEvaluation.h"

// Global variables
int seed1, seed2, seed3, seed4;

int main()
{

	//String variables for data output file
	char fname[256];
	char pname[256];
	char tmpstr[256];
	char version[256];
	double tmpparam[256];
	char paramnames[256][256];

	//Simulation Parameters from text file
	ReadSimParam (pname, tmpstr, version, tmpparam, paramnames);

	int param_cnt = 0;
	const int molecules = (const int) tmpparam[param_cnt]; param_cnt++; //# molecules []
	double alpha1; //angle between the normal to the tube exit surface and the extension directed to the substrate origin []
	double alpha2; //angle between the normal to the substrate surface and the extension between tube center and substrate origin []
	double dist;  //substrate distance [R]
	int config_num = (int) tmpparam[param_cnt]; param_cnt++; //numbers of incidence set
	double *alpha1Array = new double[config_num];
	double *alpha2Array = new double[config_num];
	double *distArray = new double[config_num];
	for (int i = 0; i < config_num; i++) {
		alpha1Array[i] = tmpparam[param_cnt]; param_cnt++;
		alpha2Array[i] = tmpparam[param_cnt]; param_cnt++;
		distArray[i] = tmpparam[param_cnt]; param_cnt++;
	}
	const double tubelength = tmpparam[param_cnt]; param_cnt++; //tube length [R]
	const double planecenter = tubelength; //on axis position of plane cutting the tube [R]
	const double planetilt = tmpparam[param_cnt]; param_cnt++; //tilt of plane cutting the tube []
	const double outerradius = tmpparam[param_cnt]; param_cnt++; //outer radius of tube [R]
	bool hole =  (tmpparam[param_cnt]==1); param_cnt++;
	const double z_hole = tmpparam[param_cnt]; param_cnt++; //z position of "hole-"cylinder axis crossing the tube axis [R]
	const double tilt_hole = tmpparam[param_cnt]; param_cnt++; //tilt of "hole-"cylinder axis relative to tube axis []
	const double r_hole = tmpparam[param_cnt]; param_cnt++; //radius of "hole-"cylinder [R]
	const double TC_hole = tmpparam[param_cnt]; param_cnt++; //Transmission coefficient of "hole-"cylinder [R]
	bool freeflight = (tmpparam[param_cnt]==1); param_cnt++;
	const double lambda_mean = tmpparam[param_cnt]; param_cnt++; //mean free path [R]
	bool pressure_decay = (tmpparam[param_cnt]==1); param_cnt++;
	const double DeltaP = tmpparam[param_cnt]; param_cnt++; //total pressure decay between entry and exit tube surface [%]
	const double uptakecoeff = tmpparam[param_cnt]; param_cnt++; //uptake coefficient per tube collision! (overall uptake coefficient = TC/(TR+BS+UT)*uptakecoeff)
	const int gridsize = (const int) tmpparam[param_cnt]; param_cnt++; //# of pixels (1D) of the mapping area
	const double imagesize = tmpparam[param_cnt]; param_cnt++; //physical image area [R]
	int virtual_no = (int) tmpparam[param_cnt]; param_cnt++; //Number of virtual planes (including entrance and exit plane)
	int site[3];
	site[0] = (int)tmpparam[param_cnt]+(int)(gridsize/2); param_cnt++; //x site [in grid pixels] of angular distribution determination
	site[1] = (int)tmpparam[param_cnt]+(int)(gridsize/2); param_cnt++; //y site [in grid pixels] of angular distribution determination
	site[2] = (int)tmpparam[param_cnt]; param_cnt++; //neighbourhood [in grid pixels] of angular distribution determination
	int HistoRes = (int)tmpparam[param_cnt]; param_cnt++; //angular resolution of histogram []
	seed1 = (int) tmpparam[param_cnt]; param_cnt++;
	seed2 = (int) tmpparam[param_cnt]; param_cnt++;
	seed3 = (int) tmpparam[param_cnt]; param_cnt++;
	seed4 = (int) tmpparam[param_cnt];
	int total_param_cnt = param_cnt+1;

	WriteSummary (pname, tmpstr, version, tmpparam, paramnames, total_param_cnt);
	cout << "* * * * * * * * * * *" << endl << endl;

	//Flags
	bool endflag;
	bool col = false;
	bool col_old;
	bool on_site;

	//Case variable
	//Last Interaction Flag (LIF)
	//Cases (0: Last Molecule Collision (LMC), 1: Last Tube Collision (LTC), 2: Direct Flight Through (DFT))
	int LIF;

	//Current molecule variables
	valarray<double> position(3);
	valarray<double> position_old(3);

	valarray<double> direction(3);
	valarray<double> direction_new(3);
	valarray<double> position_substrate(3);
	valarray<double> direction_substrate(3);
	double path; //Path since last molecular collision
	double path_tot; //Total path transmitted
	double lambda; //Mean Free Path following the logarithmic law

	//Counters
	double TC = 0; //Tube Collision
	double MC = 0; //Molecule Collision
	double UT = 0; //UpTake
	double BS = 0; //BackScattered
	double TR = 0; //TRansmitted
	double TRH = 0; //TRansmitted through Hole
	//TRansmitted (Direct FlighT, Last Molecule Collision, Last Tube Collision)
	double TR_DFT = 0, TR_LMC = 0, TR_LTC = 0;

	//Counter Arrays: 1st dimension: configuration
	valarray<double> LIS(30); //Lost In Space
	valarray<double> LOS(30); //Lost On Substrate (besides the mapped area)
	valarray<double> DB(30);  //Desorption Back
	for (int k = 0; k < 30; k++) {
		LIS[k] = 0;
		LOS[k] = 0;
		DB[k] = 0;
	}

	//Mapping data containers
	valarray< valarray< valarray<int> > > mStrike; //1st dimension: x/2nd dimension: y/3d dimension: configuration
	int mSize_x = gridsize;
	int mSize_y = gridsize;
	mInit(mStrike, mSize_x, mSize_y);

	//Data container for angular distribution evaluation at deposition site
	valarray< valarray<int> > substrate_zenith;
	valarray< valarray<int> > substrate_azimuth;
	substrate_zenith.resize (3);
	substrate_azimuth.resize (3);
	for (int k = 0; k < 3; k++) {
		substrate_zenith[k].resize(360/HistoRes);
		substrate_azimuth[k].resize(360/HistoRes);
		for (int i = 0; i < 360/HistoRes; i++) {
			substrate_zenith[k][i] = 0;
			substrate_azimuth[k][i] = 0;
		}
	}

	//Tube surface collision distribution
	double *TubeCollCounter = new double[virtual_no];

	//Pressure inside tube
	double *MFPsampling = new double[virtual_no];
	double *MFPsectionCount = new double[virtual_no];
	double *virtual_pos = new double[virtual_no];
	for (int i = 0; i < virtual_no; i++) {
		MFPsampling[i] = 0;
		MFPsectionCount[i] = 0;
		virtual_pos[i] = i*(planecenter + tan(degree2rad(planetilt)))/(virtual_no-1);
	}
	double *TRflux = new double[virtual_no];
	double *BSflux = new double[virtual_no];

	//Collision distribution
	double *CollisionCounter = new double[virtual_no];

	//Timer initialization
	double dif = -1;
	double dif_old = -1;
	double togo_old = -1;
	time_t start, end;
	time (&start);

	strcpy (fname, pname);
	ofstream OutFile1; strcat (fname, "Progress.txt"); OutFile1.open(fname);
	strcpy (fname, pname);
	OutFile1.precision(6);

	/*########################*/

	// Update simulation if continued.
	strcpy (fname, pname);
	strcat (fname, "Statistics_export_Config1.txt");
	if (FileExists(fname)) {

		cout << "The simulation " << pname << " is continued." << endl;
		UpdateDataContainers (pname, config_num, mStrike,
						   substrate_zenith, substrate_azimuth,
						   virtual_no, TRflux, BSflux, MFPsampling,
						   MFPsectionCount, TubeCollCounter, CollisionCounter, TC,
						   MC, UT, BS, TR, TRH, TR_LMC, TR_LTC,
						   TR_DFT, LIS, LOS, DB);
	}

	/*########################*/


	time_t rawtime;
	struct tm * timeinfo;
	time (&rawtime);
	timeinfo = localtime (&rawtime);
	OutFile1 << "Start time: " << asctime (timeinfo);

	int perc = 5;
	int mol = 0;
	int site_cnt = 0;

	/*######################## while all molecules ###################################*/

	while (mol < molecules) {

		MolInit(position, direction, path_tot, path, lambda_mean, lambda);
		LIF = 2;

		/************************* while in tube **************************************/
		endflag = false;
		while (!endflag) {

			position_old = position;

			MolTube (position, direction, direction_new, path_tot, path);

			col_old = col;
			col = false;
			if(!freeflight) {
//				col = true: The molecule undergoes a molecule collision (however it might be outside the tube)

				if(pressure_decay) {
					MFP_adapt (lambda, lambda_mean, path, position, position_old, virtual_pos, virtual_no, MFPsampling, MFPsectionCount,
						direction, planecenter, planetilt, DeltaP);
					col = MolMol_pressure_decay (position, direction, direction_new, path_tot, path, lambda_mean, lambda, planecenter, DeltaP);
				}
				else col = MolMol (position, direction, direction_new, path_tot, path, lambda_mean, lambda, planecenter);

			}

			if (MolBack (position, direction)) {
				endflag = true;
				BS++; //The molecule is backscattered
			}
			else if (MolTransTubeEnd (position, direction, path_tot, planecenter, planetilt)) {
				endflag = true;

				TR++; //The molecule has transmitted the tube

				mol++; //The molecule leaves the tube

				//Elapsed time in percentage (seconds)
				if ((100*(mol+1))/molecules > perc) {
					time (&end);
					dif_old = dif;
					dif = difftime (end,start);
					OutFile1 << perc << " % (" << dif-dif_old << " s)" << endl;
					if (dif-dif_old == 0 || togo_old == floor ((dif-dif_old)/5*(100-perc)/60)) {
						cout << perc << " % done" << endl;
					}
					else {
						togo_old = floor ((dif-dif_old)/5*(100-perc)/60);

						if (floor ((dif-dif_old)/5*(100-perc)/60) == 0) cout << perc << " % done: < 1 min to go" << endl;
						else cout << perc << " % done: " << floor ((dif-dif_old)/5*(100-perc)/60) << " min to go" << endl;
					}

					perc = perc+5;
				}

				switch (LIF) {
					case 0: TR_LMC++; MC++; break;
					case 1: TR_LTC++; TC++; break;
					case 2: TR_DFT++; break;
				}

				// loop through all the substrate configs defined by the alpha1, alpha2 and Ls
				for (int i = 0; i < config_num; i++) {
					alpha1 = alpha1Array[i];
					alpha2 = alpha2Array[i];
					dist = distArray[i];
					int index = i;

					if (SubstrateStrike (position, direction, position_substrate, direction_substrate, path_tot, alpha1, alpha2, dist, planecenter, planetilt)) {
						//The molecule arrives on the substrate

						Desorption_Back (position_substrate, DB, alpha1, alpha2, dist, planecenter, planetilt, index);

						if (SubstrateMappingArray (position_substrate, imagesize, mStrike, index, site, on_site)) {
							//mol++; //The molecule arrives in the mapped zone
							if (on_site) { //for the extraction of an angular probability distribution at a certain area
							//if (LIF == 0) { //for the distinction of last interaction angular distribution
								GetAngularDistribution (site_cnt, direction_substrate, substrate_zenith, substrate_azimuth, HistoRes, LIF);
							}
						}
						else LOS[index]++;
					}
					else LIS[index]++;
				}
			}
			else {
				//The molecule is still inside the tube
				direction = direction_new;

				if (col) {
					LIF = 0;
					MC++; //The molecule collides with a molecule inside the tube

					GetCollisionDistribution (position, CollisionCounter, virtual_no, planecenter, planetilt);

				}
				else if (hole && MolTransHole (position, z_hole, tilt_hole, r_hole, TC_hole)) {
					endflag = true;
					TRH++; //The molecule is transmitted through the hole
				}
				else {
					LIF = 1;
					TC++; //The molecule collides with the tube

					GetTubeSurfaceDistribution (position, TubeCollCounter, virtual_no, planecenter, planetilt);

					if (MolUptake (uptakecoeff)) {
						endflag = true;
						UT++; //The molecule sticks on the tube wall
					}

				}

			}

			MolPressure (position, position_old, virtual_pos, virtual_no, TRflux, BSflux);
		}
		/************************* while in tube **************************************/
	}

	/*######################## while all molecules ###################################*/

	cout << 100 << " % done" << endl;

	//Elapsed time
	time (&end);
	dif = difftime (end,start);
	if (floor (dif/60) == 0) cout << endl << "Elapsed time: " << dif << " s" << endl;
	else cout << endl << "Elapsed time: " << floor (dif/60) << " min " << (int)dif % 60 << " s" << endl;

	//Data Output to text file
	cout << endl << "Saved text files:" << endl;

	MappingDataOutput (pname, config_num, mStrike, imagesize, mSize_x, mSize_y, TR);

	MappingAngularDistribution (pname, substrate_zenith, substrate_azimuth, HistoRes);

	MappingPressureDataOutput (pname, virtual_pos, virtual_no, TRflux, BSflux, TR, TRH, BS, UT, MFPsampling, MFPsectionCount, TubeCollCounter, CollisionCounter);

	//Write Simulation Parameter file
	WriteStatisticsTube (pname, TC, MC, UT, BS, TR, TRH, TR_LMC, TR_LTC, TR_DFT, dif);
	WriteStatisticsSubstrate (pname, config_num, UT, BS, TR, TRH, LIS, LOS, DB, dif);
	WriteStatisticsExport (pname, config_num, TC, MC, UT, BS,TR, TRH, TR_LMC, TR_LTC, TR_DFT, LIS, LOS, DB);

	GetConfigurationPlotData (pname, imagesize, planecenter, planetilt, alpha1Array, alpha2Array, distArray, config_num, hole, r_hole, z_hole, tilt_hole, outerradius);

	//End time
	time (&rawtime);
	timeinfo = localtime (&rawtime);
	OutFile1 << "End time: " << asctime (timeinfo);

	OutFile1.close();

	delete[] distArray;
	delete[] alpha1Array;
	delete[] alpha2Array;
	delete[] TubeCollCounter;
	delete[] MFPsampling;
	delete[] MFPsectionCount;
	delete[] virtual_pos;
	delete[] TRflux;
	delete[] BSflux;
	delete[] CollisionCounter;

	cout << endl << "C++ simulation finished on " << asctime (timeinfo) <<  endl;
	cout << "***************************************************" << endl << endl;

	return 0;
}
